<?php

namespace App\Services;
use QuickBooksOnline\API\DataService\DataService;
use App\Models\Customer;

class QuickBooksService
{
    public static function checkQuickBooksSession()
    {
        $accessToken = session('qb_access_token');
        $realmId = session('qb_company_id');
        if (!$accessToken || !$realmId) {
            return false;
        }
        return true;
    }

    public static function getDataService($requireSession = true){
        if ($requireSession) {
            $accessToken = session('qb_access_token');
            $refreshToken = session('qb_refresh_token');
            $realmId = session('qb_company_id');

            if (!$accessToken || !$realmId) {
                throw new \Exception('QuickBooks session is missing. Please reconnect.');
            }

            $dataService = DataService::Configure([
                'auth_mode' => 'oauth2',
                'ClientID' => config('services.quickbooks.client_id'),
                'ClientSecret' => config('services.quickbooks.client_secret'),
                'RedirectURI' => config('services.quickbooks.redirect_uri'),
                'scope' => 'com.intuit.quickbooks.accounting',
                'baseUrl' => config('services.quickbooks.env') === 'sandbox'
                    ? "Development"
                    : "Production",
                'QBORealmID' => $realmId,
                'accessTokenKey' => $accessToken,
                'refreshTokenKey' => $refreshToken,
            ]);

            // Refresh tokens if needed
            $oauthHelper = $dataService->getOAuth2LoginHelper();
            $accessTokenObj = $oauthHelper->refreshAccessTokenWithRefreshToken($refreshToken);
            session([
                'qb_access_token' => $accessTokenObj->getAccessToken(),
                'qb_refresh_token' => $accessTokenObj->getRefreshToken()
            ]);
            $dataService->updateOAuth2Token($accessTokenObj);
            return $dataService;

        } else {
            // Used during callback BEFORE access token is saved
            return DataService::Configure([
                'auth_mode' => 'oauth2',
                'ClientID' => config('services.quickbooks.client_id'),
                'ClientSecret' => config('services.quickbooks.client_secret'),
                'RedirectURI' => config('services.quickbooks.redirect_uri'),
                'scope' => 'com.intuit.quickbooks.accounting',
                'baseUrl' => config('services.quickbooks.env') === 'sandbox'
                    ? "Development"
                    : "Production",
            ]);
        }
    }

    public static function refreshAccessToken($refreshToken)
{
    $dataService = DataService::Configure([
        'auth_mode'       => 'oauth2',
        'ClientID'        => config('services.quickbooks.client_id'),
        'ClientSecret'    => config('services.quickbooks.client_secret'),
        'RedirectURI'     => config('services.quickbooks.redirect_uri'),
        'scope'           => 'com.intuit.quickbooks.accounting',
        'baseUrl'         => config('services.quickbooks.env') === 'sandbox' ? 'Development' : 'Production',
        'refreshTokenKey' => $refreshToken,
    ]);

    $oauthHelper = $dataService->getOAuth2LoginHelper();

    try {
        $refreshedToken = $oauthHelper->refreshAccessTokenWithRefreshToken($refreshToken);

        // Optional: update session or save to DB
        session([
            'qb_access_token'     => $refreshedToken->getAccessToken(),
            'qb_refresh_token'    => $refreshedToken->getRefreshToken(),
            'qb_token_expires_at' => now()->addSeconds(3600)
        ]);

        // Update internal token in SDK
        $dataService->updateOAuth2Token($refreshedToken);

        return $refreshedToken;
    } catch (\Exception $e) {
        \Log::error('QuickBooks token refresh failed: ' . $e->getMessage());
        throw new \Exception('Unable to refresh token.');
    }
}


    public static function syncCustomers(){
        $dataService = self::getDataService();
        $startPosition = 1;
        $maxResults = 100;

        do {
            $query = "SELECT * FROM Customer STARTPOSITION $startPosition MAXRESULTS $maxResults";
            $customers = $dataService->Query($query);

            if (!$customers || count($customers) === 0) {
                break;
            }

            foreach ($customers as $qbCustomer) {
                Customer::updateOrCreate(
                    ['qb_id' => $qbCustomer->Id],
                    [
                        'display_name' => $qbCustomer->DisplayName,
                        'email' => $qbCustomer->PrimaryEmailAddr->Address ?? null,
                        'phone' => $qbCustomer->PrimaryPhone->FreeFormNumber ?? null,
                        'billing_address' => json_encode($qbCustomer->BillAddr ?? []),
                    ]
                );
            }

            $startPosition += $maxResults;

        } while (count($customers) === $maxResults);
    }
}
