<?php

namespace App\Http\Controllers;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use App\Services\QuickBooksService;
use QuickBooksOnline\API\Facades\Invoice;
use QuickBooksOnline\API\Facades\Payment;
use QuickBooksOnline\API\Core\OAuth\OAuth2\OAuth2AccessToken;
use App\Models\Invoice as InvoiceModel;
class QuickBooksController extends Controller
{
    public function connect(){
    try {
        $dataService = QuickBooksService::getDataService();
        // Check if access token exists in session
        if (!session()->has('qb_access_token') || !session('qb_access_token')) {
            throw new \Exception("Missing token");
        }
        // If everything is fine, redirect to dashboard or intended route
       // return redirect()->route('quickbooks.dashboard');
        return redirect('/invoices');
    } catch (\Exception $e) {
        // When tokens are missing or expired, redirect to get fresh OAuth token
        $dataService = \QuickBooksOnline\API\DataService\DataService::Configure([
            'auth_mode'       => 'oauth2',
            'ClientID'        => config('services.quickbooks.client_id'),
            'ClientSecret'    => config('services.quickbooks.client_secret'),
            'RedirectURI'     => config('services.quickbooks.redirect_uri'),
            'scope'           => 'com.intuit.quickbooks.accounting',
            'baseUrl'         => config('services.quickbooks.env') === 'sandbox' ? 'Development' : 'Production',
        ]);

        $oauthHelper = $dataService->getOAuth2LoginHelper();
        $authUrl = $oauthHelper->getAuthorizationCodeURL();
        return redirect($authUrl);
    }
}

public function callback(Request $request)
{
    if (! $request->has(['code', 'realmId'])) {
        return redirect()->route('quickbooks.connect')
                        ->withErrors('Missing authorization code or realmId.');
    }

    try {
        // Don't require session tokens during initial callback
        $dataService = QuickBooksService::getDataService(false);

        $oauthHelper = $dataService->getOAuth2LoginHelper();
        $accessTokenObj = $oauthHelper->exchangeAuthorizationCodeForToken(
            $request->code,
            $request->realmId
        );

        session([
            'qb_access_token'  => $accessTokenObj->getAccessToken(),
            'qb_refresh_token' => $accessTokenObj->getRefreshToken(),
            'qb_company_id'    => $request->realmId,
        ]);

        $dataService->updateOAuth2Token($accessTokenObj);

        return redirect()->route('invoices.index')
                        ->with('success', 'Connected to QuickBooks!');
    } catch (\Exception $e) {
        \Log::error('QuickBooks OAuth callback failed: '.$e->getMessage());
        return redirect()->route('quickbooks.connect')
                        ->withErrors('QuickBooks authentication failed. Please try again.');
    }
}

    public function syncCustomers(){
        // dd(Session::all());
        // exit;
        // $allTokens = QuickBooksService::refreshAccessToken("RT1-151-H0-1754068785wfi6idpjoqqtket9exco");
        //  dd($allTokens);
        // exit;
        if (!QuickBooksService::checkQuickBooksSession()) {
            return response()->json(['status'=>'error','msg' => 'QuickBooks session is missing. Please reconnect.'], 200);
        }
        try {
            QuickBooksService::syncCustomers();
            $customers = DB::table('customers')->pluck('display_name','qb_id')->toArray();
            //convert to select option html
            $options = '<option value="">Select Customer</option>';
            foreach ($customers as $key => $value) {
                $options .= '<option value="'.$key.'">'.$value.'</option>';
            }
            return response()->json(['status'=>'succ','customers'=>$options,'msg' => 'Customers synced successfully']);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    public function createInvoice(Request $req){  
        if (!QuickBooksService::checkQuickBooksSession()) {
            return response()->json(['status'=>'error','msg' => 'QuickBooks session is missing. Please reconnect.'], 200);
        }
        $dataService = QuickBooksService::getDataService();
        $dataService->setLogLocation(storage_path('logs/qb.log'));
        $dataService->throwExceptionOnError(true);

        $invoiceIds = explode(',', $req->id);
        $createdInvoices = [];
        $invoices = InvoiceModel::with('items')->whereIn('id', $invoiceIds)->where('qb_export_status', '!=', 1)->get();
        foreach ($invoices as $invoice) {
            $customerId = $invoice->merchant ?? null;
            if (!$customerId) {
                logger()->warning("Invoice {$invoice->id} skipped: missing customerId.");
                continue;
            }
    
            $qbLineItems = [];
            foreach ($invoice->items as $item) {
                $qbLineItems[] = [
                    "DetailType" => "SalesItemLineDetail",
                    "Amount" => $item->quantity * $item->amount,
                    "Description" => $item->description,
                    "SalesItemLineDetail" => [
                        "ItemRef" => [
                            "name" => $item->description,
                            "value" => 1,
                        ],
                        "UnitPrice" => $item->amount,
                        "Qty" => $item->quantity,
                    ]
                ];
            }
    
            $billingAddress = [
                "Line1" => $invoice->address ?? '',
            ];
    
            $qbInvoice = [
                "CustomerRef" => ["value" => $customerId],
                "DocNumber" => $invoice->invoice_id,
                "TxnDate" => $invoice->invoicedate,
                "DueDate" => $invoice->duedate,
                "BillAddr" => $billingAddress,
                "Line" => $qbLineItems,
            ];
    
            $invoiceObject = Invoice::create($qbInvoice);
            $result = $dataService->Add($invoiceObject);
            // echo '<pre>';
            // print_r($result);
          //  exit;
            if ($error = $dataService->getLastError()) {
                logger()->error("Invoice {$invoice->id} failed: " . $error->getResponseBody());
            } else {
                $createdInvoices[] = [
                    'local_id' => $invoice->id,
                    'qb_id' => $result->Id,
                ];
                InvoiceModel::where('id', $invoice->id)->update([
                    'qb_invoice_id' => $result->Id,
                    'qb_export_status'=>1
                ]);
                // Optional: update the invoice record
                // $invoice->update(['quickbooks_invoice_id' => $result->Id]);
            }
        }
        return response()->json(['status'=>'succ','invoices'=>$createdInvoices,'msg' => 'Invoices exported successfully'], 200);
    }

    public function invoicePayment(){
        // You already have this
        
        $dataService = QuickBooksService::getDataService(); 
        $invoice = $dataService->FindById("Invoice", 155);
       // dd($invoice);
        
        $customer = $dataService->FindById("Customer", 1);
        // dd($customer->Active); // Should be true
        // exit;
        $paymentData = Payment::create([
            "CustomerRef" => [
                "value" => 1 // same as used in invoice
            ],
            "TotalAmt" => 30.00, // the amount you're receiving
            "Line" => [
                [
                    "Amount" => 30.00,
                    "LinkedTxn" => [
                        [
                            "TxnId" => 155, // QuickBooks Invoice ID
                            "TxnType" => "Invoice"
                        ]
                    ]
                ]
            ]
        ]);
        
        $result = $dataService->Add($paymentData);
        
        if ($error = $dataService->getLastError()) {
            dd("Error adding payment: " . $error->getResponseBody());
        } else {
            dd("Payment added successfully! Payment ID: " . $result->Id);
        }
        
    }
    
    public function updateInvoice(Request $req){
        $dataService = QuickBooksService::getDataService();
        $dataService->setLogLocation(storage_path('logs/qb.log'));
        $dataService->throwExceptionOnError(true);

         $quickbooksInvoiceId = 10136;
        $invoice = $dataService->FindById('Invoice', $quickbooksInvoiceId);
        if ($invoice) {
            // Step 2: Modify fields
            $invoice->DueDate = '2025-04-30'; // or any field you want to update

            // You can also replace line items, like this:
            $invoice->Line = [
                [
                    "DetailType" => "SalesItemLineDetail",
                    "Amount" => 100.00,
                    "Description" => "Updated Description",
                    "SalesItemLineDetail" => [
                        "ItemRef" => [
                            "value" => "1", // Must be valid Item ID in QBO
                        ],
                        "UnitPrice" => 100,
                        "Qty" => 1
                    ]
                ]
            ];

            // Step 3: Update it in QuickBooks
            $updatedInvoice = $dataService->Update($invoice);

            if ($error = $dataService->getLastError()) {
                dd('Update Failed: ' . $error->getResponseBody());
            } else {
                dd('Invoice updated successfully. ID: ' . $updatedInvoice->Id);
            }
        } else {
            dd("Invoice not found.");
        }
    }

    public function uploadPdf()
    {
        $dataService = QuickBooksService::getDataService();

        $accessToken = new OAuth2AccessToken(
            session('qb_access_token'),
            '',
            session('qb_refresh_token'),
            ''
        );

        $dataService->updateOAuth2Token($accessToken);
        $dataService->throwExceptionOnError(true);

        $pdfPath = storage_path('app/invoice.pdf');
        $fileName = 'MyLaravelInvoice.pdf';

        if (!file_exists($pdfPath)) {
            return response()->json(['error' => 'PDF file not found.'], 404);
        }

        $fileData = file_get_contents($pdfPath);

        $uploadResult = $dataService->Upload([
            'FileName' => $fileName,
            'ContentType' => 'application/pdf',
            'FileBytes' => $fileData
        ], true); // true = attach to last created entity

        return response()->json(['message' => 'PDF uploaded', 'result' => $uploadResult]);
    }
}
