<?php
namespace App\Http\Controllers\Auth;
use Illuminate\Support\Facades\Auth;
use App\Http\Requests;
use Illuminate\Http\Request;
use Response;
use App\Http\Controllers\Controller;
use App\Providers\RouteServiceProvider;
use App\Models\User;
use Illuminate\Foundation\Auth\RegistersUsers;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
class RegisterController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Register Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users as well as their
    | validation and creation. By default this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
    */

    use RegistersUsers;

    /**
     * Where to redirect users after registration.
     *
     * @var string
     */
    protected $redirectTo = RouteServiceProvider::HOME;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest');
    }

    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array  $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function validator(array $data)
    {
        return Validator::make($data, [
            'name' => ['required', 'string', 'max:255'],
            'captcha' => ['captcha','string', 'max:50'],
            'contact_number' => ['required', 'string', 'max:20'],
            'referral' => ['required', 'string', 'max:50'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
        ]);
    }

    public function getPendingParent($parent_id){
        $childUsers  = User::where(['parent_id'=>$parent_id])->get();
        $child  = 0;
        foreach($childUsers as $childUser){
          $count  = User::where(['parent_id'=>$childUser->id])->count();
          if($count<get_settings('caping_limit')){
            $child  =  $childUser->id;
            break;
          }
        }
        return $child;
    }

    public function register(Request $req){
          $this->form_validation($req);
          $count  = User::where(['parent_id'=>$req->referral])->count();
          if($count<get_settings('caping_limit')){
            $parent_id  = $req->referral;
          } else {
            $parent_id  = $this->getPendingParent($req->referral);
          }
         $user  = User::create([
             'name' => $req->name,
             'email' => $req->email,
             'parent_id'=>$parent_id,
             'role'=>'user',
             'status'=>'pending',
             'contact_number' => $req->contact_number,
             'referral' => $req->referral,
             'password' => Hash::make($req->password),
         ]);
         $credentials = $req->only('email', 'password');
         if(Auth::attempt($credentials)){
           return response(['status' => 'succ','msg' => 'Registered successfully']);
         }
      }

      public function form_validation($r){
        $rules = [
             'name' => 'required|min:2|max:100',
             'agreeTerms'=>'required',
             'email' => 'required|unique:users,email',
              'referral' => [
                'required',
                Rule::exists('users', 'id')
                ->where(function ($query){
                    $query->where('status', 'active');
                }),
              ],
             'contact_number' => 'required|unique:users,contact_number',
             'captcha'     => 'required|captcha',
       ];

         $rules['password'] = 'required';
         $rules['password_confirmation'] = 'required|same:password';
         $customMessages = [
             'username.unique' => 'The username has already been taken. Please Login',
             'captcha.captcha' => 'Captcha is invalid',
             'email.unique' => 'The email has already been taken. Please Login',
             'mobile_no.unique' => 'The Mobile Number has already been taken. Please Login',
             'referral.exists'=>'The entered referral is invalid.',
         ];
         $r->validate($rules,$customMessages);
      }


    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     * @return \App\User
     */
    protected function create(array $data)
    {
        $parent_id  = User::where(['email'=>$data['referral']])->value('parent_id');
        return User::create([
            'name' => $data['name'],
            'email' => $data['email'],
            'parent_id'=>$parent_id,
            'role'=>'user',
            'status'=>'deactive',
            'contact_number' => $data['contact_number'],
            'referral' => $data['referral'],
            'password' => Hash::make($data['password']),
        ]);
    }

    public function refreshCaptcha(){
      echo json_encode(array('img'=>captcha_src()));
    }


}
